package gov.va.med.mhv.usermgmt.service.impl;

import gov.va.med.mhv.usermgmt.bizobj.EmployeeBO;
import gov.va.med.mhv.usermgmt.bizobj.EmployeeOrganizationRoleAssembler;
import gov.va.med.mhv.usermgmt.bizobj.EmployeeOrganizationRoleBO;
import gov.va.med.mhv.usermgmt.bizobj.EmployeeOrganizationRoleHistoryAssembler;
import gov.va.med.mhv.usermgmt.bizobj.EmployeeOrganizationRoleHistoryBO;
import gov.va.med.mhv.usermgmt.bizobj.OrganizationAssembler;
import gov.va.med.mhv.usermgmt.bizobj.OrganizationBO;
import gov.va.med.mhv.usermgmt.bizobj.RoleAssembler;
import gov.va.med.mhv.usermgmt.bizobj.RoleBO;
import gov.va.med.mhv.usermgmt.enumeration.OrganizationTypeEnumeration;
import gov.va.med.mhv.usermgmt.enumeration.RoleActionEnumeration;
import gov.va.med.mhv.usermgmt.messages.EmployeeManagementMessages;
import gov.va.med.mhv.usermgmt.messages.UserManagementMessages;
import gov.va.med.mhv.usermgmt.service.EmployeeOrganizationRoleCollectionServiceResponse;
import gov.va.med.mhv.usermgmt.service.EmployeeOrganizationRoleHistoryCollectionServiceResponse;
import gov.va.med.mhv.usermgmt.service.EmployeeOrganizationRoleServiceResponse;
import gov.va.med.mhv.usermgmt.service.EmployeeRoleManagementService;
import gov.va.med.mhv.usermgmt.service.EmployeeSearchResultServiceResponse;
import gov.va.med.mhv.usermgmt.service.EmployeeServiceResponse;
import gov.va.med.mhv.usermgmt.service.OrganizationCollectionServiceResponse;
import gov.va.med.mhv.usermgmt.service.RoleCollectionServiceResponse;
import gov.va.med.mhv.usermgmt.transfer.Employee;
import gov.va.med.mhv.usermgmt.transfer.EmployeeOrganizationRole;
import gov.va.med.mhv.usermgmt.transfer.EmployeeOrganizationRoleHistory;
import gov.va.med.mhv.usermgmt.transfer.Organization;
import gov.va.med.mhv.usermgmt.transfer.Role;
import gov.va.med.mhv.usermgmt.transfer.TransferObjectFactory;

import java.util.Date;
import java.util.List;

import org.apache.commons.lang.StringUtils;
import org.tigris.atlas.service.AbstractService;

/**
 * Service implementation class for the EmployeeRoleManagement service
 */
public class EmployeeRoleManagementServiceImpl  extends AbstractService  implements EmployeeRoleManagementService {
	public gov.va.med.mhv.usermgmt.service.EntityMaintenanceService getEntityService() {
		return gov.va.med.mhv.usermgmt.service.ServiceFactory.createEntityMaintenanceService();
	}
	public gov.va.med.mhv.usermgmt.service.EmployeeSearchService getSearchService() {
		return gov.va.med.mhv.usermgmt.service.ServiceFactory.createEmployeeSearchService();
	}

	/**
	 * Execute the UpdateEmployeeOrganizationRole service
	 *
	 * @return EmployeeOrganizationRoleServiceResponse
	 */
	public EmployeeOrganizationRoleServiceResponse updateEmployeeOrganizationRole(Employee requestingEmployee, EmployeeOrganizationRole employeeOrgRole) {
		EmployeeOrganizationRoleServiceResponse response = new EmployeeOrganizationRoleServiceResponse();

		//outer if statement to ensure user has access to even attempt to edit this role
		if(getEmployeeOrganizationRolesEmployeeCanManage(requestingEmployee).getEmployeeOrganizationRoles().contains(employeeOrgRole)){

			//attempt to save entity
			response = getEntityService().save(employeeOrgRole);

			//check for errors, and if not present, setup and save the role history entry
			if(!response.getMessages().hasErrorMessages() && !response.getEmployeeOrganizationRole().getAllMessages().hasErrorMessages()){
				EmployeeOrganizationRoleHistory employeeOrganizationRoleHistory = TransferObjectFactory.createEmployeeOrganizationRoleHistory();
				employeeOrganizationRoleHistory.setEmployeeOrganizationRole(employeeOrgRole);
				employeeOrganizationRoleHistory.setPerformingEmployee(requestingEmployee);
				employeeOrganizationRoleHistory.setModificationDate(new Date());
				if(employeeOrgRole.getActive()){
					employeeOrganizationRoleHistory.setRoleAction(RoleActionEnumeration.getEnum(RoleActionEnumeration.ACTIVATE));
				}else{
					employeeOrganizationRoleHistory.setRoleAction(RoleActionEnumeration.getEnum(RoleActionEnumeration.DEACTIVATE));
				}
				response.getMessages().addMessages(getEntityService().save(employeeOrganizationRoleHistory).getMessages());
			}
		} else {
			addError(response, EmployeeManagementMessages.MANAGEROLES_SECURITY_ERROR, null);
		}

		if(!response.getMessages().hasErrorMessages() && !response.getEmployeeOrganizationRole().getAllMessages().hasErrorMessages()){
			addInfo(response, EmployeeManagementMessages.MANAGEROLES_EDIT_SUCCESS,
					new String[]{ employeeOrgRole.getEmployee().getFirstName(), employeeOrgRole.getEmployee().getLastName(), employeeOrgRole.getOrganization().getName(), employeeOrgRole.getRole().getName() });
		}
		return response;
	}

	/**
	 * Execute the AssignEmployeeOrganizationRole service
	 *
	 * @return EmployeeOrganizationRoleServiceResponse
	 */
	public EmployeeOrganizationRoleServiceResponse assignEmployeeOrganizationRole(Employee requestingEmployee, String employeeUserName, Organization org, Role role) {
		EmployeeOrganizationRoleServiceResponse response = new EmployeeOrganizationRoleServiceResponse();

		EmployeeOrganizationRole newEmpOrgRole = TransferObjectFactory.createEmployeeOrganizationRole();
		//outer if statement to ensure requesting employee has permissions to perform this role assignment
		if(getRolesAtOrganizationEmployeeCanManage(requestingEmployee, org).getRoles().contains(role)){
			newEmpOrgRole.setEmployee( saveEmployeeLocally(employeeUserName));
			newEmpOrgRole.setActive(true);
			newEmpOrgRole.setOrganization(org);
			newEmpOrgRole.setRole(role);
			response = getEntityService().save(newEmpOrgRole);

			//check for errors, and if not present, setup and save the role history entry
			if(!response.getMessages().hasErrorMessages() && !response.getEmployeeOrganizationRole().getAllMessages().hasErrorMessages()){
				EmployeeOrganizationRoleHistory employeeOrganizationRoleHistory = TransferObjectFactory.createEmployeeOrganizationRoleHistory();
				employeeOrganizationRoleHistory.setEmployeeOrganizationRole(response.getEmployeeOrganizationRole());
				employeeOrganizationRoleHistory.setPerformingEmployee(requestingEmployee);
				employeeOrganizationRoleHistory.setRoleAction(RoleActionEnumeration.getEnum(RoleActionEnumeration.ADD));
				employeeOrganizationRoleHistory.setModificationDate(new Date());
				response.getMessages().addMessages(getEntityService().save(employeeOrganizationRoleHistory).getMessages());
			}
		} else {
			addError(response, EmployeeManagementMessages.MANAGEROLES_SECURITY_ERROR, null);
		}

		if(!response.getMessages().hasErrorMessages() && !response.getEmployeeOrganizationRole().getAllMessages().hasErrorMessages()){
			addInfo(response, EmployeeManagementMessages.MANAGEROLES_ASSIGN_SUCCESS,
					new String[]{ newEmpOrgRole.getEmployee().getFirstName(),newEmpOrgRole.getEmployee().getLastName(), newEmpOrgRole.getOrganization().getName(), newEmpOrgRole.getRole().getName() });
		}
		return response;
	}

	/**
	 * Execute the GetEmployeeByUserName service
	 * VA Employees can exist in Active Directory, and locally in our internal DB
	 * If an employee is found in AD, but not yet saved locally, this method will return
	 * the employee entity, not in a persistant state.
	 * @return EmployeeServiceResponse
	 */
	public EmployeeServiceResponse getEmployeeByUserName(String userName) {
		EmployeeServiceResponse response = new EmployeeServiceResponse();
		if(StringUtils.isNotEmpty(userName)){
			Employee employee;
			userName = userName.toLowerCase().trim();

			//search for employee
			List results = EmployeeBO.getEmployeeByUserName(userName);

			//if valid results found, get that previously saved employee entity
			if (results != null && results.size() > 0 && !((EmployeeBO)results.get(0)).getAllMessages().hasErrorMessages()){
				employee = ((EmployeeBO)results.get(0)).getEmployeeValues();
			} else {
				employee = TransferObjectFactory.createEmployee();
			}

			//fetch employee from active directory
			EmployeeSearchResultServiceResponse employeeSearchResult = getSearchService().searchByUserName(userName);

			//if employee not in our environment and also not found in active directory, error
			if(employee.getEmployeePK() == null && employeeSearchResult.getEmployeeSearchResult() == null){
				addError( response, UserManagementMessages.USER_DOES_NOT_EXIST, new String[] { userName } );
			} else{
				//else, assuming all is well, update employee entity with most recent data from active directory
				employee.setUserName(userName);
				if(employeeSearchResult.getEmployeeSearchResult() != null){
					employee.setFirstName(employeeSearchResult.getEmployeeSearchResult().getFirstName());
					employee.setLastName(employeeSearchResult.getEmployeeSearchResult().getLastName());
				}
				response.setEmployee(employee);
			}
		}
		return response;
	}

	/**
	 * Save employee record into local environment
	 *
	 * @return EmployeeServiceResponse
	 */
	private Employee saveEmployeeLocally(String userName) {
		return getEntityService().save(getEmployeeByUserName(userName).getEmployee()).getEmployee();
	}

	/**
	 * Execute the GetEmployeeOrganizationRolesAssignedToEmployee service
	 *
	 * @return EmployeeOrganizationRoleCollectionServiceResponse
	 */
	public EmployeeOrganizationRoleCollectionServiceResponse getEmployeeOrganizationRolesAssignedToEmployee(Employee employee) {
		EmployeeOrganizationRoleCollectionServiceResponse response = new EmployeeOrganizationRoleCollectionServiceResponse();
		if(employee != null && employee.getId() != null){
			List results = EmployeeOrganizationRoleBO.getEmployeeOrganizationRolesAssignedToEmployee(employee.getId());
			if(results != null && !results.isEmpty()){
				response.addEmployeeOrganizationRoles(EmployeeOrganizationRoleAssembler.getEmployeeOrganizationRoleCollection(results));
			}
		}
		return response;
	}

	/**
	 * Execute the GetRoleHistoriesForEmployee service
	 *
	 * @return EmployeeOrganizationRoleHistoryCollectionServiceResponse
	 */
	public EmployeeOrganizationRoleHistoryCollectionServiceResponse getRoleHistoriesForEmployee(Employee employee) {
		EmployeeOrganizationRoleHistoryCollectionServiceResponse response = new EmployeeOrganizationRoleHistoryCollectionServiceResponse();
		if(employee != null && employee.getId() != null){
			List results = EmployeeOrganizationRoleHistoryBO.getRoleHistoriesForEmployee(employee.getId());
			if(results != null && !results.isEmpty()){
				response.addEmployeeOrganizationRoleHistorys(EmployeeOrganizationRoleHistoryAssembler.getEmployeeOrganizationRoleHistoryCollection(results));
			}
		}
		return response;
	}

	/**
	 * Execute the GetOrganizationsEmployeeCanManage service
	 *
	 * @return OrganizationTypeDefCollectionServiceResponse
	 */
	public OrganizationCollectionServiceResponse getOrganizationsEmployeeCanManage(Employee employee) {
		OrganizationCollectionServiceResponse response = new OrganizationCollectionServiceResponse();
		if(employee != null && employee.getId() != null){
			List results = OrganizationBO.getOrganizationsEmployeeCanManage(employee.getId());
			if(results != null && !results.isEmpty()){
				response.addOrganizations(OrganizationAssembler.getOrganizationCollection(results));
			}
		}
		return response;
	}

	/**
	 * Execute the GetRolesAtOrgEmployeeCanManage service
	 *
	 * @return RoleCollectionServiceResponse
	 */
	public RoleCollectionServiceResponse getRolesAtOrganizationEmployeeCanManage(Employee employee, Organization organization) {
		RoleCollectionServiceResponse response = new RoleCollectionServiceResponse();
		if(employee != null && employee.getId() != null && organization != null){
			List results =  RoleBO.getRolesAtOrganizationEmployeeCanManage(employee.getId(),organization.getId());
			if(results != null && !results.isEmpty()){
				response.addRoles(RoleAssembler.getRoleCollection(results));
			}
		}
		return response;
	}

	/**
	 * Execute the GetEmployeeOrganizationRolesEmployeeCanManage service
	 *
	 * @return EmployeeOrganizationRoleCollectionServiceResponse
	 */
	public EmployeeOrganizationRoleCollectionServiceResponse getEmployeeOrganizationRolesEmployeeCanManage(Employee employee) {
		EmployeeOrganizationRoleCollectionServiceResponse response = new EmployeeOrganizationRoleCollectionServiceResponse();
		if(employee != null && employee.getId() != null){
			List results =  EmployeeOrganizationRoleBO.getEmployeeOrganizationRolesEmployeeCanManage(employee.getId());
			if(results != null && !results.isEmpty()){
				response.addEmployeeOrganizationRoles(EmployeeOrganizationRoleAssembler.getEmployeeOrganizationRoleCollection(results));
			}
		}
		return response;
	}

	/**
	 * Execute the GetOrganizationsOfTypeWhereEmployeeHasRole service
	 *
	 * @return OrganizationCollectionServiceResponse
	 */
	public OrganizationCollectionServiceResponse getOrganizationsOfTypeWhereEmployeeHasRole(Employee employee, OrganizationTypeEnumeration orgType, String roleName) {
		OrganizationCollectionServiceResponse response = new OrganizationCollectionServiceResponse();
		if(employee != null && employee.getId() != null && orgType != null && StringUtils.isNotEmpty(roleName)){
			List results =  OrganizationBO.getOrganizationsOfTypeWhereEmployeeHasRole(employee.getId(), orgType, roleName);
			if(results != null && !results.isEmpty()){
				response.addOrganizations(OrganizationAssembler.getOrganizationCollection(results));
			}
		}
		return response;
	}

	/**
	 * Execute the GetRolesByName service
	 *
	 * @return RoleCollectionServiceResponse
	 */
	public RoleCollectionServiceResponse getRolesByName(String roleName) {
		RoleCollectionServiceResponse response = new RoleCollectionServiceResponse();
		List results = RoleBO.getRoleByName(roleName);
		if(results != null && !results.isEmpty()){
			response.addRoles(RoleAssembler.getRoleCollection(results));
		}
		return response;
	}
}